import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Scissors, User, Calendar, Image as ImageIcon, Package, CheckCircle, Award, Heart } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const navigate = useNavigate()
  const [courses, setCourses] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('sewingCourses')
    const defaultCourses = [
      {
        id: 1,
        project: 'Платье-футляр',
        teacher: 'Елена Иванова',
        completionDate: '2025-05-15',
        status: 'active',
        resultPhoto: null
      },
      {
        id: 2,
        project: 'Блузка с воротником',
        teacher: 'Мария Петрова',
        completionDate: '2025-04-20',
        status: 'completed',
        resultPhoto: 'https://images.unsplash.com/photo-1594633313593-bab3825d0caf?w=400'
      },
      {
        id: 3,
        project: 'Юбка-карандаш',
        teacher: 'Анна Соколова',
        completionDate: '2025-03-10',
        status: 'completed',
        resultPhoto: 'https://images.unsplash.com/photo-1594633313593-bab3825d0caf?w=400'
      }
    ]
    
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        const validCourses = parsed.filter(course => 
          course.project && course.teacher && course.completionDate &&
          course.completionDate.match(/^\d{4}-\d{2}-\d{2}$/)
        )
        if (validCourses.length > 0) {
          setCourses(parsed)
        } else {
          setCourses(defaultCourses)
          localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
        }
      } catch (e) {
        setCourses(defaultCourses)
        localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
      }
    } else {
      setCourses(defaultCourses)
      localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
    }
  }, [])

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  return (
    <div className="my-courses-page-modern">
      <div className="container">
        <section className="page-header-modern">
          <div className="header-badge-modern">
            <Scissors size={18} />
            <span>ВАШИ ПРОЕКТЫ</span>
          </div>
          <h1 className="page-title-modern">
            Мои <span className="title-accent-modern">курсы</span>
          </h1>
          <p className="page-description-modern">
            Отслеживайте прогресс ваших проектов и делитесь результатами
          </p>
        </section>

        <ImageSlider />

        <PortfolioSection />

        {courses.length === 0 ? (
          <div className="empty-state-modern">
            <div className="empty-icon-wrapper-modern">
              <Package size={64} />
            </div>
            <h2 className="empty-title-modern">Нет курсов</h2>
            <p className="empty-text-modern">Запишитесь на первый курс по шитью</p>
            <button 
              className="empty-button-modern" 
              onClick={() => navigate('/enroll')}
            >
              Записаться на курс
            </button>
          </div>
        ) : (
          <section className="courses-list-modern">
            <div className="courses-section-header-modern">
              <h2 className="courses-section-title-modern">Мои записи на курсы</h2>
              <p className="courses-section-description-modern">
                Отслеживайте прогресс ваших проектов и делитесь результатами
              </p>
            </div>
            <div className="courses-grid-modern">
              {courses.map((course) => (
                <CourseCard
                  key={course.id}
                  course={course}
                  formatDate={formatDate}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const CourseCard = ({ course, formatDate }) => {
  return (
    <div className={`course-item-modern ${course.status}`}>
      <div className="course-header-modern">
        <div className="course-icon-wrapper-modern">
          <Scissors size={28} />
        </div>
        <div className="course-info-modern">
          <div className="course-number-modern">#{course.id}</div>
          <div className={`status-badge-modern ${course.status}`}>
            {course.status === 'active' ? (
              <>
                <CheckCircle size={14} />
                <span>В процессе</span>
              </>
            ) : (
              <>
                <Award size={14} />
                <span>Завершен</span>
              </>
            )}
          </div>
        </div>
      </div>
      
      <h3 className="course-title-modern">{course.project}</h3>
      
      <div className="course-details-modern">
        <div className="detail-item-modern">
          <User size={16} />
          <div>
            <span className="detail-label-modern">Преподаватель:</span>
            <span className="detail-value-modern">{course.teacher}</span>
          </div>
        </div>
        <div className="detail-item-modern">
          <Calendar size={16} />
          <div>
            <span className="detail-label-modern">Дата завершения:</span>
            <span className="detail-value-modern">{formatDate(course.completionDate)}</span>
          </div>
        </div>
      </div>
      
      {course.status === 'completed' && course.resultPhoto && (
        <div className="course-result-modern">
          <div className="result-photo-wrapper-modern">
            <img 
              src={course.resultPhoto} 
              alt={course.project}
              className="result-photo-modern"
              onError={(e) => {
                e.target.style.display = 'none'
                e.target.nextSibling.style.display = 'flex'
              }}
            />
            <div className="result-photo-placeholder-modern" style={{ display: 'none' }}>
              <ImageIcon size={48} />
            </div>
          </div>
          <div className="result-label-modern">
            <ImageIcon size={14} />
            <span>Фото результата</span>
          </div>
        </div>
      )}
    </div>
  )
}

const PortfolioSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const portfolioItems = [
    {
      title: 'Платье-футляр',
      category: 'Повседневная одежда',
      image: 'linear-gradient(135deg, #FF6B6B 0%, #4ECDC4 100%)',
      likes: 124
    },
    {
      title: 'Блузка с воротником',
      category: 'Офисный стиль',
      image: 'linear-gradient(135deg, #9B59B6 0%, #FF6B6B 100%)',
      likes: 98
    },
    {
      title: 'Юбка-карандаш',
      category: 'Классика',
      image: 'linear-gradient(135deg, #4ECDC4 0%, #FFA726 100%)',
      likes: 156
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`portfolio-section-modern ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="portfolio-header-modern">
          <h2 className="portfolio-title-modern">Портфолио работ</h2>
          <p className="portfolio-description-modern">
            Примеры изделий наших студентов
          </p>
        </div>
        <div className="portfolio-grid-modern">
          {portfolioItems.map((item, index) => (
            <div 
              key={index} 
              className="portfolio-item-modern"
              style={{ 
                animationDelay: `${index * 0.15}s`,
                background: item.image
              }}
            >
              <div className="portfolio-overlay-modern">
                <div className="portfolio-content-modern">
                  <div className="portfolio-category-modern">{item.category}</div>
                  <h3 className="portfolio-item-title-modern">{item.title}</h3>
                  <div className="portfolio-likes-modern">
                    <Heart size={18} />
                    <span>{item.likes}</span>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

export default MyCourses

